/**
 * Dynamic API route handler for FileMaker OData
 * Simplified to only use Host 1
 */

import { NextRequest, NextResponse } from 'next/server';
import { getHost, getTable } from '@/lib/config/databases';
import { createODataClient } from '@/lib/api/odataClient';
import { checkEnvironmentVariables } from '@/lib/utils/envCheck';

// Log environment variables on server side
function logEnvVars() {
  console.log('API Route - Server-side Environment Variables');
  console.log('FILEMAKER_ODATA_URL1:', process.env.FILEMAKER_ODATA_URL1 ? 'SET' : 'NOT SET');
  console.log('FILEMAKER_USERNAME1:', process.env.FILEMAKER_USERNAME1 ? 'SET' : 'NOT SET');
  console.log('FILEMAKER_PASSWORD1:', process.env.FILEMAKER_PASSWORD1 ? 'SET' : 'NOT SET');
  console.log('NEXT_PUBLIC_API_URL:', process.env.NEXT_PUBLIC_API_URL || 'NOT SET');
}

export async function GET(request: NextRequest) {
  const params = request.nextUrl.pathname.split('/').slice(3);
  console.log('GET request received:', request.url);
  
  try {
    // Extract path from URL pathname
    const path = params;
    console.log('Path params:', path);
    
    // Log the full URL and path for debugging
    console.log('Full request URL:', request.url);
    
    console.log('Path length:', path.length);
    if (path.length >= 3) {
      console.log('Database:', path[0]);
      console.log('Table:', path[1]);
      console.log('ID:', path[2]);
    }
    
    if (!path || path.length < 2) {
      return NextResponse.json(
        { error: 'Invalid path. Format should be /api/odata/[databaseName]/[tableName]/[id?] or /api/odata/[databaseName]/Script.[scriptName]' },
        { status: 400 }
      );
    }

    // Always use host1
    const hostId = 'host1';
    
    // Extract path components
    const [databaseName, ...restPath] = path;
    let tableOrScript = restPath[0];
    let recordId = restPath[1];
    
    // Check if the table name contains parentheses for ID (e.g., "Clients(29)")
    const tableWithIdRegex = /^([^(]+)\((\d+|'[^']*')\)$/;
    const match = tableOrScript ? tableOrScript.match(tableWithIdRegex) : null;
    
    if (match) {
      console.log('Found OData format with ID in parentheses');
      tableOrScript = match[1]; // Extract the table name
      recordId = match[2].replace(/^'|'$/g, ''); // Remove quotes if present
      console.log(`Extracted table: ${tableOrScript}, ID: ${recordId}`);
    }

    // Log environment variables
    logEnvVars();
    
    // Check if this is a script execution request (starts with 'Script.')
    const isScriptExecution = tableOrScript && tableOrScript.startsWith('Script.');
    if (isScriptExecution) {
      const scriptName = tableOrScript.replace('Script.', '');
      console.log(`Script execution requested: ${scriptName}`);
      
      // Construct the script URL in the format: https://db-data.net/fmi/odata/v4/TrackPad-FMS-test/Script.odata_test
      const odataUrl = process.env.FILEMAKER_ODATA_URL1 || '';
      // Remove any trailing slashes and append the script path
      const baseUrl = odataUrl.endsWith('/') ? odataUrl.slice(0, -1) : odataUrl;
      // The odataUrl already includes the database name, so we don't need to add it again
      const scriptUrl = `${baseUrl}/Script.${scriptName}${request.nextUrl.search}`;
      
      console.log('=== Script Execution Details ===');
      console.log('Base OData URL:', odataUrl);
      console.log('Database Name:', databaseName);
      console.log('Script Name:', scriptName);
      console.log('Final Script URL:', scriptUrl);
      console.log('===============================');
      
      // Forward the request to the FileMaker server
      const response = await fetch(scriptUrl, {
        method: 'GET',
        headers: {
          'Authorization': request.headers.get('Authorization') || `Basic ${Buffer.from(
            `${process.env.FILEMAKER_USERNAME1}:${process.env.FILEMAKER_PASSWORD1}`
          ).toString('base64')}`,
          'Content-Type': 'application/json',
        },
      });
      
      if (!response.ok) {
        const errorText = await response.text();
        console.error(`Script execution failed: ${response.status} - ${errorText}`);
        return NextResponse.json(
          { error: `Script execution failed: ${response.status} - ${errorText}` },
          { status: response.status }
        );
      }
      
      try {
        const data = await response.json();
        return NextResponse.json(data);
      } catch (error) {
        console.error('Error parsing script response:', error);
        return NextResponse.json(
          { error: 'Error parsing script response' },
          { status: 500 }
        );
      }
    }
    
    // Create OData client for Host 1
    const odataClient = createODataClient(hostId);
    
    // Handle record retrieval by ID
    if (recordId) {
      const result = await odataClient.getRecordById(databaseName, tableOrScript, recordId);
      return NextResponse.json(result);
    }
    
    // Handle table query - use the raw query string from the URL
    const rawQueryString = request.url.split('?')[1] || '';
    console.log('Raw query string:', rawQueryString);
    
    // Extract the search parameters from the URL
    const searchParams = request.nextUrl.searchParams;
    
    // Create a clean object for non-OData parameters (like _t timestamp)
    const queryParams: Record<string, string> = {};
    
    // Extract OData parameters ($filter, $select, etc.) and other parameters separately
    let odataParams = '';
    
    // Process each parameter
    searchParams.forEach((value, key) => {
      if (key.startsWith('$')) {
        // For OData parameters, collect them in the odataParams string
        // This preserves the exact format needed by OData
        if (odataParams) odataParams += '&';
        odataParams += `${key}=${value}`;
        console.log(`OData parameter ${key}:`, value);
      } else if (key.includes(' eq ') || key.includes(' ne ') || 
                key.includes(' gt ') || key.includes(' lt ') || 
                key.includes(' ge ') || key.includes(' le ')) {
        // This is likely an OData filter expression without the $filter prefix
        // Add it to the OData parameters with the proper prefix
        if (odataParams) odataParams += '&';
        odataParams += `$filter=${key}`;
        console.log(`Detected OData filter expression:`, key);
      } else {
        // Regular parameters
        queryParams[key] = value;
      }
    });
    
    // Log the parameters for debugging
    console.log('OData parameters:', odataParams);
    console.log('Other parameters:', queryParams);
    
    // Pass the raw OData parameters string to the OData client
    // This ensures the OData syntax is preserved exactly as needed
    // Add the raw OData parameters to the query params object
    if (odataParams) {
      queryParams._odataRawParams = odataParams;
    }
    const result = await odataClient.getRecords(databaseName, tableOrScript, queryParams);
    
    // Log the result structure
    console.log('Result structure:', {
      hasValue: result.hasOwnProperty('data'),
      dataLength: result.data ? result.data.length : 0,
      count: result.count
    });
    return NextResponse.json(result);
  } catch (error: any) {
    console.error('OData API error:', error);
    return NextResponse.json(
      { error: error.message || 'Internal Server Error' },
      { status: error.status || 500 }
    );
  }
}

export async function POST(request: NextRequest) {
  try {
    // Extract path from URL pathname
    const path = request.nextUrl.pathname.split('/').slice(3);
    
    if (!path || path.length < 2) {
      return NextResponse.json(
        { error: 'Invalid path. Format should be /api/odata/[databaseName]/[tableName] or /api/odata/[databaseName]/Script.[scriptName]' },
        { status: 400 }
      );
    }
    
    // Always use host1
    const hostId = 'host1';
    
    const [databaseName, tableOrScript] = path;
    const isScriptExecution = tableOrScript.startsWith('Script.');
    
    // Handle script execution
    if (isScriptExecution) {
      const scriptName = tableOrScript.replace('Script.', '');
      console.log(`Script execution (POST) requested: ${scriptName}`);
      
      // Construct the script URL in the format: https://db-data.net/fmi/odata/v4/TrackPad-FMS-test/Script.odata_test
      const odataUrl = process.env.FILEMAKER_ODATA_URL1 || '';
      const baseUrl = odataUrl.endsWith('/') ? odataUrl.slice(0, -1) : odataUrl;
      const scriptUrl = `${baseUrl}/Script.${scriptName}`;
      
      console.log('=== Script Execution (POST) Details ===');
      console.log('Script URL:', scriptUrl);
      console.log('Script Name:', scriptName);
      console.log('Host ID:', hostId);
      
      // Get request body for script parameters
      let requestBody: Record<string, any> = {};
      const contentType = request.headers.get('content-type');
      
      if (contentType && contentType.includes('application/json')) {
        try {
          requestBody = await request.json() as Record<string, any>;
        } catch (error) {
          console.log('No request body or invalid JSON');
        }
      }
      
      // Prepare the request options
      const requestOptions: RequestInit = {
        method: 'POST',
        headers: {
          'Authorization': request.headers.get('Authorization') || `Basic ${Buffer.from(
            `${process.env.FILEMAKER_USERNAME1}:${process.env.FILEMAKER_PASSWORD1}`
          ).toString('base64')}`,
          'Content-Type': 'application/json',
        },
      };
      
      // Only include body if there are parameters
      if (Object.keys(requestBody).length > 0) {
        // If scriptParameterValue is already set, use it directly
        // Otherwise, stringify the entire request body
        const scriptParam = 'scriptParameterValue' in requestBody 
          ? String(requestBody.scriptParameterValue)
          : JSON.stringify(requestBody);
          
        requestOptions.body = JSON.stringify({
          scriptParameterValue: scriptParam
        });
      }
      
      console.log('Script Request:', {
        url: scriptUrl,
        method: 'POST',
        headers: requestOptions.headers,
        body: requestOptions.body ? JSON.parse(requestOptions.body.toString()) : 'No body'
      });
      
      try {
        // Forward the request to the FileMaker server
        const response = await fetch(scriptUrl, requestOptions);
        
        if (!response.ok) {
          const errorText = await response.text();
          console.error(`Script execution failed: ${response.status} - ${errorText}`);
          return NextResponse.json(
            { error: `Script execution failed: ${response.status} - ${errorText}` },
            { status: response.status }
          );
        }
        
        try {
          const data = await response.json();
          return NextResponse.json(data);
        } catch (error) {
          console.error('Error parsing script response:', error);
          return NextResponse.json(
            { error: 'Error parsing script response' },
            { status: 500 }
          );
        }
      } catch (error) {
        console.error('Error executing script:', error);
        return NextResponse.json(
          { error: 'Error executing script' },
          { status: 500 }
        );
      }
    }
    
    // Handle regular table operations
    const table = getTable(hostId, databaseName, tableOrScript);
    if (!table) {
      return NextResponse.json(
        { error: `Table '${tableOrScript}' not found in database '${databaseName}'` },
        { status: 404 }
      );
    }
    
    // Get request body
    const data = await request.json();
    
    // Create OData client for the specified host
    const odataClient = createODataClient(hostId);
    
    // Create record
    const result = await odataClient.createRecord(databaseName, tableOrScript, data);
    
    return NextResponse.json(result, { status: 201 });
  } catch (error: any) {
    console.error('OData API error:', error);
    return NextResponse.json(
      { error: error.message || 'Internal Server Error' },
      { status: error.status || 500 }
    );
  }
}

export async function PATCH(request: NextRequest) {
  try {
    // Extract path from URL pathname
    const path = request.nextUrl.pathname.split('/').slice(3);
    
    if (!path || path.length < 3) {
      return NextResponse.json(
        { error: 'Invalid path. Format should be /api/odata/[databaseName]/[tableName]/[id]' },
        { status: 400 }
      );
    }
    
    // Always use host1
    const hostId = 'host1';
    
    const [databaseName, tableName, id] = path;
    
    // Validate host and table
    const host = getHost(hostId);
    if (!host) {
      return NextResponse.json(
        { error: `Host configuration not found` },
        { status: 404 }
      );
    }
    
    const table = getTable(hostId, databaseName, tableName);
    if (!table) {
      return NextResponse.json(
        { error: `Table '${tableName}' not found in database '${databaseName}'` },
        { status: 404 }
      );
    }
    
    // Get request body
    const data = await request.json();
    
    // Create OData client for the specified host
    const odataClient = createODataClient(hostId);
    
    // Update record
    const result = await odataClient.updateRecord(databaseName, tableName, id, data);
    
    return NextResponse.json(result);
  } catch (error: any) {
    console.error('OData API error:', error);
    return NextResponse.json(
      { error: error.message || 'Internal Server Error' },
      { status: error.status || 500 }
    );
  }
}

export async function DELETE(request: NextRequest) {
  try {
    // Extract path from URL pathname
    const path = request.nextUrl.pathname.split('/').slice(3);
    
    if (!path || path.length < 3) {
      return NextResponse.json(
        { error: 'Invalid path. Format should be /api/odata/[databaseName]/[tableName]/[id]' },
        { status: 400 }
      );
    }
    
    // Always use host1
    const hostId = 'host1';
    
    const [databaseName, tableName, id] = path;
    
    // Validate host and table
    const host = getHost(hostId);
    if (!host) {
      return NextResponse.json(
        { error: `Host configuration not found` },
        { status: 404 }
      );
    }
    
    const table = getTable(hostId, databaseName, tableName);
    if (!table) {
      return NextResponse.json(
        { error: `Table '${tableName}' not found in database '${databaseName}'` },
        { status: 404 }
      );
    }
    
    // Create OData client for the specified host
    const odataClient = createODataClient(hostId);
    
    // Delete record
    await odataClient.deleteRecord(databaseName, tableName, id);
    
    return new NextResponse(null, { status: 204 });
  } catch (error: any) {
    console.error('OData API error:', error);
    return NextResponse.json(
      { error: error.message || 'Internal Server Error' },
      { status: error.status || 500 }
    );
  }
}
